'''
Popuplib backwards compatibility module for popuplib2.
'''
import warnings

import es
import playerlib
import popuplib2

# for name-to-popup lookups
named_popups = {}

# registering construct functions
named_constructs = {}

_usermanager = popuplib2._usermanager

info = es.AddonInfo()
info.name = 'popuplib wrapper for popuplib2'
info.version = popuplib2.__version__
info.author = 'GODJonez'
info.url = 'http://python.eventscripts.com/pages/Popuplib2'
info.description = 'Backwards-compatibility layer for popuplib2'


# decorators
def _multipopup(func):
    '''
    A decorator function that takes care of iterable popup name parameters.
    '''
    def mp_func(*args, **kw):
        args = list(args)
        kw = dict(kw)
        if 'popup_name' in kw:
            name = kw['popup_name']
        else:
            name = args.pop(0)
        if isinstance(name, basestring):
            return func(name, *args, **kw)
        else:
            return [func(n, *args, **kw) for n in name]
    return mp_func

def _multiuser(func):
    '''
    A decorator function that takes care of single or multiple userids.
    '''
    def mp_func(*args, **kw):
        args = list(args)
        kw = dict(kw)
        if 'users' in kw:
            users = kw['users']
            del kw['users']
        else:
            users = args.pop()
        try:
            userid = int(users)
        except TypeError:
            pass
        else:
            users = users,
        for userid in users:
            func(users=int(userid), *args, **kw)
    return mp_func

def pos_kw_wrapper(func, gen_info):
    '''
    A wrapper function to call function with conditional
    positional/keyword arguments.
    '''
    def mp_func(args, kw, **overwrite):
        for keyword, info in gen_info.iteritems():
            value = overwrite[keyword]
            if info[0]:
                args[info[1]] = value
            else:
                kw[keyword] = value
        return func(*args, **kw)
    return mp_func
                
def _multilanguage(text_pos, lang_pos):
    '''
    A decorator function generator that takes care of single or multiple languages.
    '''
    def generate(func):
        def mp_func(*args, **kw):
            args = list(args)
            kw = dict(kw)
            if 'text' in kw:
                text = kw['text']
                del kw['text']
                text_in_place = False
            else:
                text = args[text_pos]
                text_in_place = True
            if 'lang' in kw:
                lang = kw['lang']
                del kw['lang']
                lang_in_place = False
            else:
                try:
                    lang = args[lang_pos]
                    lang_in_place = True
                except IndexError:
                    lang = None
                    lang_in_place = False
            f = pos_kw_wrapper(func,
                {
                    'text': (text_in_place, text_pos),
                    'lang': (lang_in_place, lang_pos),
                }
            )
            if isinstance(text, dict):
                # multiple languages, use dictionary:
                for lang, text in text.iteritems():
                    f(args, kw, text=text, lang=lang)
            else:
                # use as provided
                f(args, kw, text=text, lang=lang)
        return mp_func
    return generate

def _name_args(*names):
    '''
    A decorator function generator to convert args to keywords.

    Example:

    @_name_args('a', 'b')
    def test(b, a):
        return b > a

    now decorated call test(1, 2) would be the same as normal call test(a=1, b=2),
    or normal call test(2, 1) in this case.
    '''
    def generate(func):
        def mp_func(*args, **kw):
            nkw = {}
            args_gen = iter(args)
            for name in names:
                if name in kw:
                    nkw[name] = kw[name]
                else:
                    nkw[name] = args_gen.next()
            func(**nkw)
        return mp_func
    return generate

# fake classes
def _p2inst_property(new_name, description=None):
    '''Attribute that can be given directly to popuplib2 instances.'''
    def setter(self, attribute, value):
        setattr(self._p2inst, new_name, value)
    def getter(self, attribute):
        return getattr(self._p2inst, new_name)
    return property(setter, getter, None, description)

class Popup_base(object):
    '''
    Named base popup object.
    '''
    enablekeys = _p2inst_property("enable_keys", "numbers that trigger menuselect")
    
    def __init__(self, name):
        self._p2inst = popuplib2.PopupGroup()
        self.name = name
        self.prepuser = None
        self.editlang = "default"
        self._warned = False
    
    def delete(self):
        delete(self.name)
    
    def timeout(self, mode, time):
        if not self._warned:
            warnings.warn("timeout is not implemented in this version")
            self._warned = True
    
    @_multiuser
    def send(self, users):
        if self.prepuser:
            self.prepuser(users, self.name)
        self._p2inst.send(users)

    @_multiuser
    def unsend(self, users):
        self._p2inst.unsend(users)
    
    @_multiuser
    def update(self, users):
        # difference from original implementation: update view for any popup
        user = _usermanager[users]
        user.refresh()

    def __repr__(self):
        return '%s(%s)'%(
            self.__class__.__name__,
            repr(self.name),
        )


class Popup(Popup_base):
    '''
    Named basic popup.
    '''
    def __init__(self, name):
        Popup_base.__init__(self, name)
        self._p2inst.menuselect = self.handle_choice
        self.check_language(self.editlang)
        # backwards compatibility cakes:
        self.menuselect = None
        self.menuselectfb = None
        self.p_select = {}
        self.p_submenu = {}
        self.menuvar = {}
        self.menuval = {}
        # no cache... might break some very rare addons...
    
    def check_language(self, language):
        if language not in self._p2inst:
            if language == 'default':
                self._p2inst[language] = popuplib2.Popup()
            else:
                self._p2inst[language] = popuplib2.Popup(self._p2inst['default'])
    
    @_multilanguage(text_pos=1, lang_pos=2)
    def addline(self, text, lang):
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        self._p2inst[lang].append(text)

    def delline(self, line, lang=None):
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        del self._p2inst[lang][line-1]

    @_multilanguage(text_pos=2, lang_pos=3)
    def insline(self, line, text, lang):
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        self._p2inst[lang].insert(line-1, text)

    @_multilanguage(text_pos=2, lang_pos=3)
    def modline(self, line, text, lang):
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        self._p2inst[lang][line-1] = text

    def addlineAll(self, text):
        for p in self._p2inst.itervalues():
            p.append(text)

    def dellineAll(self, line):
        for p in self._p2inst.itervalues():
            del p[line-1]

    def inslineAll(self, line, text):
        for p in self._p2inst.itervalues():
            p.insert(line-1, text)

    def modlineAll(self, line, text):
        for p in self._p2inst.itervalues():
            p[line-1] = text
    
    def select(self, item, block):
        self.p_select[item] = block

    def submenu(self, item, menu):
        if menu:
            self.p_submenu[item] = menu
        elif item in self.p_submenu:
            del self.p_submenu[item]

    def menuvalue(self, variable_name, item, variable_value):
        # backwards compatibility, allow swapping name and item:
        if not isinstance(item, int):
            item, variable_name = int(variable_name), str(item)
        self.menuvar[item] = variable_name
        self.menuval[item] = variable_value
    
    def handle_choice(self, params):
        userid = params['userid']
        choice = params['choice']
        popup = params['popup']

        # check for menuvalue
        if choice in self.menuvar:
            es.set(self.menuvar[choice], self.menuval[choice])
        
        # check for select block
        pselect = self.p_select.get(choice, None)
        if pselect:
            pselect(userid, choice, self.name)

        # check for menuselect block
        if self.menuselect:
            self.menuselect(userid, choice, self.name)

        # check for menuselect fallback
        if not pselect and self.menuselectfb:
            self.menuselectfb(userid, choice, self.name)

        # check for submenu
        if choice in self.p_submenu:
            psubmenu = self.p_submenu[choice]
            if isinstance(psubmenu, basestring):
                psubmenu = named_popups[psubmenu]
            if isinstance(psubmenu, Popup_base):
                psubmenu = psubmenu._p2inst
            return psubmenu

        return None # no submenu from us


class Easymenu(Popup_base):
    '''
    Named paged menu thingy.
    '''
    _p2class = popuplib2.PagedMenu

    def __init__(self, name, variable, func):
        Popup_base.__init__(self, name)
        self._p2inst.menuselect = self.handle_choice
        self._p2inst.call_special = True
        self.check_language(self.editlang)
        self.settitle(self.name)
        # backwards compatibility cakes:
        self.p_submenu = None
        self.menuselect = None
        self.menuselectfb = func
        self.c_savevar = variable
    
    def check_language(self, language):
        if language not in self._p2inst:
            if language == 'default':
                self._p2inst[language] = self._p2class()
            else:
                default = self._p2inst['default']
                self._p2inst[language] = current = self._p2class(default)
                current.title = default.title
                current.description = default.description

    @_multilanguage(text_pos=1, lang_pos=2)
    def setdescription(self, text, lang):
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        self._p2inst[lang].description = text

    @_multilanguage(text_pos=1, lang_pos=3)
    def settitle(self, text, pagenumbers=30, lang=None):
        # difference: pagenumbers has no effect
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        self._p2inst[lang].title = text
    
    @_multilanguage(text_pos=2, lang_pos=4)
    def addoption(self, choice, text, state=True, lang=None):
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        self._p2inst[lang].add(choice, text, bool(state))

    def setoption(self, choice, text=None, state=None, lang=None):
        if not lang:
            lang = self.editlang
        if lang == "default":
            popups = self._p2inst.itervalues()
        else:
            self.check_language(lang)
            popups = self._p2inst[lang],
        found = False
        for p in popups:
            opt = p.find(choice)
            if opt:
                found = True
                if text is not None:
                    opt.text = text
                if state is not None:
                    opt.selectable = bool(state)
        if not found:
            raise KeyError(
                "Option %s does not exist in easymenu %s."%(
                    choice,
                    self.name,
                )
            ) # original was IndexError

    def submenu(self, *args):
        if len(args) not in (1,2):
            raise TypeError("Easymenu.submenu takes 1 to 2 parameters, %d given."%len(args))
        psubmenu = args[-1]
        if isinstance(psubmenu, basestring):
            psubmenu = named_popups[psubmenu]
        if isinstance(psubmenu, Popup_base):
            psubmenu = psubmenu._p2inst
        self._p2inst.previous_menu = psubmenu
    
    @_multiuser
    def _set_page_number(self, pagenumber, users):
        user = _usermanager[users]
        p = self._p2inst._get_userpopup(user)
        if 1 <= pagenumber <= p.pages():
            p.pagenum = pagenumber
        
    def sendPage(self, users, pagenumber):
        self._set_page_number(pagenumber, users)
        self.send(users)
    
    def getPage(self, userid):
        user = _usermanager[userid]
        p = self._p2inst._get_userpopup(user)
        return p.pagenum
    
    # _multiuser expects users to be the last parameter or keyword but
    # original API conflicts, use _name_args to wrap compatible API:
    @_name_args('self', 'users', 'pagenumber')
    @_multiuser
    def unsendPage(self, pagenumber, users):
        if self.getPage(users) == pagenumber:
            self.unsend(users)

    def getPageCount(self, lang=None):
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        p = self._p2inst[lang]
        return p.pages()

    def handle_choice(self, params):
        userid = params['userid']
        choice = params['choice']
        popup = params['popup']
        raw_choice = params['raw_choice']
        page = params['page']
        is_special = params['special']
        
        if self.c_savevar:
            # DO NOT LIKE
            es.set(self.c_savevar, choice or raw_choice)
        
        # check for menuselect
        if self.menuselect:
            self.menuselect(userid, choice or raw_choice, self.name)

        # check for menuselectfb
        if choice and self.menuselectfb:
            self.menuselectfb(userid, choice, self.name)


class Easylist(Easymenu):
    '''
    Named paged list thingy.
    '''
    _p2class = popuplib2.PagedList

    def __init__(self, name, data, options):
        Easymenu.__init__(self, name, None, None)
        self._p2inst.options_per_page = options
        self._p2inst[self.editlang].extend(data)

    @_multilanguage(text_pos=1, lang_pos=4)
    def additem(self, text, value=None, state=None, lang=None):
        # difference: value and state ignored
        if not lang:
            lang = self.editlang
        self.check_language(lang)
        self._p2inst[lang].append(text)

# new implementations of popuplib functions
def active(userid):
    '''
    Check which poup is active and how many are in the queue.

    Return value is a dictionary of type
    {
        'name': active_popup_name or None for popuplib2 objects,
        'count': number of active popups including current active,
        'object': active_popup_object,
    }
    Return value is None for bots.

    If the user has no active popups, count will be 0 and object None.
    '''
    user = _usermanager[int(userid)]
    if user.bot:
        return None
    if user.queue:
        count = len(user.queue)
        obj = user.queue[0]
        if hasattr(obj, 'name'):
            name = obj.name
        else:
            name = None
        return {
            'name': name,
            'count': count,
            'object': obj,
        }
    else:
        return {
            'name': '0',
            'count': 0,
            'object': None,
        }

@_multipopup
@_multiuser
def close(popup_name, users):
    '''
    Remove named popups from users without triggering menuselect 10 action.

    Parameters:
        popup_name      either a single popup name or a collection of names
        users           either a single userid or a collection of userids
    '''
    popup = named_popups[popup_name]
    popup.unsend(users)

@_multipopup
def construct(popup_name, constructor_name, flags, menuselect_func = None):
    '''
    Use a popup constructor to create a new popup.
    
    Parameters:
        popup_name          the name of the popup or a list of names
        constructor_name    the constructor to use for the popup
        flags               any specific data to give to the constructor
        menuselect_func     the function to call when an option is selected

    Returns:
        the popup instance or a list of popup instances in case a list
        of names was used
    '''
    if constructor_name in named_constructs:
        constructor = named_constructs[constructor_name]
        p = named_popups[popup_name] = constructor(
            popup_name, flags, menuselect_func
        )
        return p
    else:
        raise KeyError('There is no known constructor %s'%(
            repr(constructor_name),)
        ) # Error type is different than original

@_multipopup
def create(popup_name, cls=Popup, *args, **kw):
    '''
    Create a new popup with specified name.
    
    Parameters:
        popup_name          the name of the popup or a list of names
        cls                 the popup class (internal use only)

    Returns:
        the popup instance or a list of popup instances in case a list
        of names was used
    '''
    # a single popup name
    if popup_name in named_popups:
        old = named_popups[popup_name]
        if isinstance(old, cls):
            old.__init__(popup_name, *args, **kw)
            return old
    p = named_popups[popup_name] = cls(popup_name, *args, **kw)
    return p

@_multipopup
def delete(popup_name):
    '''
    Delete a popup with specified name.

    Parameters:
        popup_name          the name of the popup or a list of names
    '''
    del named_popups[popup_name]

@_multipopup
def easylist(popup_name, data=None, options=10):
    '''
    Display iterable data in popups, paging them as necessary.
    
    Parameters:
        popup_name          the name of the popup or a list of names
        data                the iterable data structure to display
        options             the number of items to show per page

    Returns:
        the easylist instance
    '''
    return create(popup_name, Easylist, data, options)

@_multipopup
def easymenu(popup_name, variable_name, callback):
    '''
    Create a menu with 7 options per page.

    Parameters:
        popup_name          the name of the popup or a list of names
        variable_name       the server variable to set (use None)
        callback            the function to call on choice

    Returns:
        the easymenu instance
    '''
    return create(popup_name, Easymenu, variable_name, callback)

@_multipopup
@_multiuser
def emulate(popup_name, choice, users):
    '''
    Directly cause a specific user do make a specific selection on a popup.

    This should only be used for testing as it might mess up the queuing system.
    '''
    user = _usermanager[int(users)]
    popup = named_popups[popup_name]
    warnings.warn("Do not use popuplib.emulate in any real script.")
    popup._response(user, choice)

@_multipopup
def exists(popup_name):
    '''
    Check whether the specified popup name is in use.

    Parameters:
        popup_name          the name of the popup or a list of names

    Returns:
        True or False depending on whether the name exists or not
        or a list of these values in case a list of names were supplied.
    '''
    return popup_name in named_popups

@_multipopup
def find(popup_name):
    '''
    Get popup instances based on their names or None if not found.

    Parameters:
        popup_name          the name of the popup or a list of names

    Returns:
        The popup instance or None if not found
        or a list of such return values for each name.
    '''
    return named_popups.get(popup_name, None)

@_multipopup
def isqueued(popup_name, userid):
    '''
    Find the queue indexes of popups for a specific user.

    Parameters:
        popup_name          the name of the popup or a list of names
        userid              the user whose queue to check

    Returns:
        The queue index of the named popup or None if not found
        or a list of such return values for each name.
    '''
    user = _usermanager[int(userid)]
    if not user.bot and popup_name in named_popups:
        popup = named_popups[popup_name]
        userpopup = popup._p2inst._get_userpopup(user)
        index = user.get_popup_index(userpopup)
        if index is None:
            return 0
        else:
            return index + 1
    return 0

def langstring(prefix, langobject, suffix=''):
    '''
    Helper function to add prefix and suffix to each language string.

    Parameters:
        prefix          a string to add to the beginning of each line
        langobject      a dictionary-like object with string values
        suffix          a string to add to the end of each line

    Returns:
        A dictionary with the same keys as langobject but with each value
        modified by adding prefix and suffix to them.
    '''
    new_dict = dict()
    for key, value in langobject.iteritems():
        new_dict[key] = '%s%s%s'%(
            prefix,
            value,
            suffix,
            )
    return new_dict

def quicksend(view_time, userid, text, func=None, keys='0123456789'):
    '''
    A wrapper for es.menu to use popup queue and Python callbacks.

    Parameters:
        view_time       number of seconds to show the popup (0 for infinite)
        userid          the user to receive the popup
        text            the text to display
        func            the function to call on response
        keys            the options allowed to be used
    '''
    p = popuplib2.radio.Popup((text,))
    p.enable_keys = keys
    p.menuselect = func
    if view_time:
        warnings.warn("view_time is not implemented in this version")
    p.send(int(userid))

@_multipopup
def send(popup_name, users):
    '''
    Send a named popup to users.
    '''
    # Difference: no optional prio parameter
    named_popups[popup_name].send(users)

def unsend(index, userid):
    raise NotImplementedError("This internal function is not implemented.")

unsendname = close # traditionally unsendname would trigger menuselect 10

def registerConstruct(name, function):
    '''
    Register a constructor function with a specified name.
    '''
    named_constructs[name] = function

def unregisterContstruct(name):
    '''
    Remove the named construct from being used anymore.

    Returns True if the construct was deleted, False if it was not registered.
    '''
    if name in named_constructs:
        del named_constructs[name]
        return True
    else:
        return False

# Built-in constructs

class ConstructPlayerList(Easymenu):
    '''
    Constructs a player list.
    '''
    _p2class = popuplib2.PersonalMenu

    def __init__(self, name, flags, menuselectfb):
        Easymenu.__init__(self, name, None, menuselectfb)
        self.flags = flags

    def check_language(self, language):
        if language not in self._p2inst:
            if language == 'default':
                self._p2inst[language] = self._p2class(self.rebuild)
            else:
                default = self._p2inst['default']
                self._p2inst[language] = current = self._p2class(self.rebuild, default)
                current.title = default.title
                current.description = default.description

    def rebuild(self, userid, popup):
        for player in playerlib.getPlayerList(self.flags):
            popup.add(player.userid, player.name)


registerConstruct('players', ConstructPlayerList)

